/*
 * I/O routines for Fabric Management Server
 *
 * Each fabric management server listens on one socket.  As soon
 * as a connection is established, the connecting entity sends a
 * small message defining what sort of connection is requested.
 *  FMS_CONN_FMA - an individual fabric management agent is connecting
 *  FMS_CONN_QUERY - an application wishes to make a query
 *  FMS_CONN_PEER - a connection from a peer fabric management server
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_channel.h"
#include "lf_fms_comm.h"
#include "lf_fabric.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_io.h"
#include "fms_socket.h"
#include "fms_fma.h"
#include "fms_client.h"
#include "fms_notify.h"

/*
 * static prototypes
 */
static void service_accept(struct lf_channel *chp);
static void service_error(struct lf_channel *chp);
static void new_connection_timeout(struct fms_connection *fcp);
void set_connection_type(struct lf_channel *chp);
static void connection_hangup(struct lf_channel *fcp);

/*
 * Initialize I/O related variables
 */
int
init_io_vars()
{
  int rc;

  /* allocate the structure for keeping I/O variables */
  F.io = (struct fms_io_vars *) calloc (sizeof(struct fms_io_vars), 1);
  if (F.io == NULL) LF_ERROR(("Allocating I/O vars"));

  /* defaults for I/O subsystem */
  F.io->service_socket = -1;			/* not open yet */
  F.io->service_port = FMS_SERVICE_PORT;

  rc = lf_init_channels();
  if (rc != 0) LF_ERROR(("Error initializing channels"));

  return 0;

 except:
   if (F.io != NULL) free(F.io);
   return -1;
}

/*
 * Initialize the I/O subsystem
 */
int
init_io()
{
  struct lf_channel *chp;
  
  chp = NULL;

  /* initialize the socket we listen on */
  F.io->service_socket = open_socket(F.io->service_port);
  if (F.io->service_socket == -1) LF_ERROR(("initializing service socket"));

  /* allocate and fill in a channel structure */
  chp = (struct lf_channel *) calloc(1, sizeof(struct lf_channel));
  if (chp == NULL) LF_ERROR(("allocate initial poll association"));

  chp->fd = F.io->service_socket;
  chp->hangup_rtn = service_error;

  /* register this with the I/O subsystem */
  lf_add_channel(chp);
  
  /* register for connections on this channel - a connect looks like
   * "ready for reading"
   * */
  lf_channel_receive(chp, NULL, 0, service_accept);
  return 0;

 except:
  if (chp != NULL) free(chp);
  if (F.io->service_socket != -1) close(F.io->service_socket);
  return -1;
}

/*
 * Handle a new connection on the service socket
 * Allocate a "new connection" structure and set up a timeout
 * to wait for first connection.
 */
static void
service_accept(
  struct lf_channel *scp)
{
  int ns;
  struct fms_connection *fcp;
  struct lf_channel *chp;

  /* init */
  ns = -1;		/* no socket yet */

  /* a place to hold some new connection variables */
  fcp = (struct fms_connection *) calloc(1, sizeof(struct fms_connection));
  if (fcp == NULL) LF_ERROR(("allocate new connection"));

  /* get the socket for the new connection */
  ns = new_connection(scp->fd, &fcp->addr);
  if (ns == -1) LF_ERROR(("accepting new connection"));

  /* allocate and fill in a new channel */
  chp = (struct lf_channel *) calloc(1, sizeof(struct lf_channel));
  if (chp == NULL) LF_ERROR(("allocate initial poll association"));

  chp->fd = ns;
  chp->hangup_rtn = connection_hangup;
  chp->context = fcp;

  /* register this with the I/O subsystem */
  lf_add_channel(chp);

  /* setup to receive connection type */
  lf_channel_receive(chp, &fcp->type, sizeof(fcp->type), set_connection_type);

  fcp->chp = chp;	/* record channel struct */

  /* set up a timeout for getting the connection type */
  fcp->timeout_event = lf_schedule_event(
  				(lf_event_handler_t) new_connection_timeout,
				(void *) fcp,
				FMS_NEW_CONNECT_TIMEOUT);

  return;

 except:
  fms_perror();
  if (fcp != NULL) free(fcp);
  if (ns != -1) close(ns);
  return;
}

/*
 * An error occurred on the service socket - try to re-open it
 */
void
service_error(
  struct lf_channel *chp)
{
  fprintf(stderr, "Error on service channel!\n");
  exit(1);
}

/*
 * The type field of a new connection has arrived
 */
void
set_connection_type(
  struct lf_channel *chp)
{
  struct fms_connection *fcp;
  int rc;

  /* get pointer to the connection */
  fcp = chp->context;

  /* delete the timeout event from the scheduler */
  lf_remove_event(fcp->timeout_event);


  /*
   * call the appropriate connection setup based on type
   */
  switch (fcp->type) {

  case FMS_CONN_FMA:		/* an agent */
    rc = init_fma_connection(fcp);
    if (rc == -1) {
      close_connection(fcp);
      LF_ERROR(("initializing FMA connection"));
    }
    break;

  case FMS_CONN_GENERIC:	/* a client connection */
    fms_notify(FMS_EVENT_DEBUG, "New client connection");
    rc = fms_init_client_connection(fcp);
    if (rc == -1) {
      close_connection(fcp);
      LF_ERROR(("Error initializing client connection"));
    }
    break;

  default:			/* bad type - close the connection */
    close_connection(fcp);
    LF_ERROR(("bad connection type: %d\n", fcp->type));
  }
  return;

 except:
  fms_perror();
}

/*
 * Close out a connection and free it
 */
void
close_connection(
  struct fms_connection *fcp)
{
  struct lf_channel *chp;

  chp = fcp->chp;
  lf_remove_channel(chp); 	/* take this channel out of the poll list */
  close(chp->fd);		/* close the socket */

  free(fcp->addr);
  free(fcp);			/* free the connection struct */
  free(chp);			/* and free the channel struct */
}

/*
 * A new connection hung up.
 * Remove the timeout and close the channel.
 */
static void
connection_hangup(
  struct lf_channel *chp)
{
  struct fms_connection *fcp;

  fcp = chp->context;
printf("new connection hung up w/o type\n");
  lf_remove_event(fcp->timeout_event);
  close_connection(fcp);
}

/*
 * Connection was made, but type never sent
 * Just close the channel.
 */
static void
new_connection_timeout(
  struct fms_connection *fcp)
{
printf("timeout waiting for type\n");
  close_connection(fcp);
}

/*
 * Cleanly shutdown the I/O subsystem
 */
void
close_io()
{
}
